'use strict';

/* --------------------------------------------------------------
 emails_table.js 2016-10-11
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2016 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * ## Emails Table Controller
 *
 * This controller will handle the main table operations of the admin/emails page.
 *
 * @module Controllers/emails_table
 */
gx.controllers.module('emails_table', [jse.source + '/vendor/datatables/jquery.dataTables.min.css', jse.source + '/vendor/datatables/jquery.dataTables.min.js', gx.source + '/libs/emails', 'modal', 'datatable', 'normalize'],

/** @lends module:Controllers/emails_table */

function (data) {

    'use strict';

    // ------------------------------------------------------------------------
    // VARIABLE DEFINITION
    // ------------------------------------------------------------------------

    var
    /**
     * Module Reference
     *
     * @type {object}
     */
    $this = $(this),


    /**
     * Modal Selector
     *
     * @type {object}
     */
    $modal = $('#emails-modal'),


    /**
     * Default Module Options
     *
     * @type {object}
     */
    defaults = {
        emailsTableActions: function emailsTableActions() {
            return '<div class="row-actions">' + '<span class="send-email action-item" title="' + jse.core.lang.translate('send', 'buttons') + '">' + '<i class="fa fa-envelope-o"></i>' + '</span>' + '<span class="forward-email action-item" title="' + jse.core.lang.translate('forward', 'buttons') + '">' + '<i class="fa fa-share"></i>' + '</span>' + '<span class="delete-email action-item" title="' + jse.core.lang.translate('delete', 'buttons') + '">' + '<i class="fa fa-trash-o"></i>' + '</span>' + '<span class="preview-email action-item" title="' + jse.core.lang.translate('preview', 'buttons') + '">' + '<i class="fa fa-eye"></i>' + '</span>' + '</div>';
        },

        convertPendingToString: function convertPendingToString(data, type, row, meta) {
            return data === true ? jse.core.lang.translate('email_pending', 'emails') : jse.core.lang.translate('email_sent', 'emails');
        }
    },


    /**
     * Final Module Options
     *
     * @type {object}
     */
    options = $.extend(true, {}, defaults, data),


    /**
     * Module Object
     *
     * @type {object}
     */
    module = {};

    // ------------------------------------------------------------------------
    // EVENT HANDLERS
    // ------------------------------------------------------------------------

    /**
     * Toggle row selection for main page table.
     *
     * @param {object} event Contains event information.
     */
    var _onSelectAllRows = function _onSelectAllRows(event) {
        if ($(this).prop('checked')) {
            $this.find('tbody .single-checkbox').addClass('checked');
            $this.find('tbody input:checkbox').prop('checked', true);
        } else {
            $this.find('tbody .single-checkbox').removeClass('checked');
            $this.find('tbody input:checkbox').prop('checked', false);
        }
    };

    /**
     * Will send the email to its contacts (even if its status is "sent").
     *
     * @param {object} event Contains event information.
     */
    var _onSendEmail = function _onSendEmail(event) {
        var $row = $(this).parents('tr');

        jse.libs.modal.message({
            title: jse.core.lang.translate('send', 'buttons'),
            content: jse.core.lang.translate('prompt_send_email', 'emails'),
            buttons: [{
                text: jse.core.lang.translate('no', 'lightbox_buttons'),
                click: function click() {
                    $(this).dialog('close');
                }
            }, {
                text: jse.core.lang.translate('yes', 'lightbox_buttons'),
                click: function click() {
                    var email = $row.data();
                    jse.libs.emails.sendCollection([email]).done(function (response) {
                        $this.DataTable().ajax.reload();
                        jse.libs.emails.getAttachmentsSize($('#attachments-size'));
                    }).fail(function (response) {
                        var title = jse.core.lang.translate('error', 'messages');

                        jse.libs.modal.message({
                            title: title,
                            content: response.message
                        });
                    });
                    $(this).dialog('close');
                }
            }]
        });
    };

    /**
     * Display modal with email information but without contacts.
     *
     * The user will be able to set new contacts and send the email (kind of "duplicate" method).
     *
     * @param {object} event Contains event information.
     */
    var _onForwardEmail = function _onForwardEmail(event) {
        var email = $(this).parents('tr').data();

        jse.libs.emails.resetModal($modal);
        jse.libs.emails.loadEmailOnModal(email, $modal);

        // Clear contact fields but let the rest of the data untouched.
        $modal.find('#email-id').val('');
        $modal.find('#sender-email, #sender-name').val('');
        $modal.find('#reply-to-email, #reply-to-name').val('');
        $modal.find('#recipient-email, #recipient-name').val('');
        $modal.find('#contacts-table').DataTable().clear().draw();

        $modal.dialog({
            title: jse.core.lang.translate('forward', 'buttons'),
            width: 1000,
            height: 800,
            modal: true,
            dialogClass: 'gx-container',
            closeOnEscape: false,
            buttons: jse.libs.emails.getDefaultModalButtons($modal, $this),
            open: jse.libs.emails.colorizeButtonsForEditMode
        });
    };

    /**
     * Delete selected row email.
     *
     * @param {object} event Contains event information.
     */
    var _onDeleteEmail = function _onDeleteEmail(event) {
        var $row = $(this).parents('tr'),
            email = $row.data();

        jse.libs.modal.message({
            title: jse.core.lang.translate('delete', 'buttons'),
            content: jse.core.lang.translate('prompt_delete_email', 'emails'),
            buttons: [{
                text: jse.core.lang.translate('no', 'lightbox_buttons'),
                click: function click() {
                    $(this).dialog('close');
                }
            }, {
                text: jse.core.lang.translate('yes', 'lightbox_buttons'),
                click: function click() {
                    jse.libs.emails.deleteCollection([email]).done(function (response) {
                        $this.DataTable().ajax.reload();
                        jse.libs.emails.getAttachmentsSize($('#attachments-size'));
                    }).fail(function (response) {
                        var title = jse.core.lang.translate('error', 'messages');
                        jse.libs.modal.message({
                            title: title,
                            content: response.message
                        });
                    });
                    $(this).dialog('close');
                }
            }]
        });
    };

    /**
     * Display modal with email information
     *
     * The user can select an action to perform upon the previewed email (Send, Forward,
     * Delete, Close).
     *
     * @param  {object} event Contains event information.
     */
    var _onPreviewEmail = function _onPreviewEmail(event) {
        var email = $(this).parents('tr').data();

        jse.libs.emails.resetModal($modal);
        jse.libs.emails.loadEmailOnModal(email, $modal);

        $modal.dialog({
            title: jse.core.lang.translate('preview', 'buttons'),
            width: 1000,
            height: 800,
            modal: false,
            dialogClass: 'gx-container',
            closeOnEscape: false,
            buttons: jse.libs.emails.getPreviewModalButtons($modal, $this),
            open: jse.libs.emails.colorizeButtonsForPreviewMode
        });
    };

    // ------------------------------------------------------------------------
    // INITIALIZATION
    // ------------------------------------------------------------------------

    /**
     * Initialize method of the module, called by the engine.
     *
     * The emails table operates with server processing because it is much faster and efficient than preparing
     * and sending all the records in every AJAX request. Check the Emails/DataTable controller method for
     * requested data and the following link for more info about server processing in DataTables.
     *
     * {@link http://www.datatables.net/manual/server-side}
     */
    module.init = function (done) {
        // Create a DataTable instance for the email records.
        jse.libs.datatable.create($this, {
            processing: false,
            serverSide: true,
            dom: 'rtip',
            autoWidth: false,
            language: jse.core.config.get('languageCode') === 'de' ? jse.libs.datatable.getGermanTranslation() : null,
            ajax: {
                url: jse.core.config.get('appUrl') + '/admin/admin.php?do=Emails/DataTable',
                type: 'POST'
            },
            order: [[2, 'desc']],
            pageLength: 20,
            columns: [{
                data: null,
                orderable: false,
                defaultContent: '<input type="checkbox" data-single_checkbox />',
                width: '2%',
                className: 'dt-head-center dt-body-center'
            }, {
                data: 'row_count',
                orderable: false,
                width: '3%',
                className: 'dt-head-center dt-body-center'
            }, {
                data: 'creation_date',
                width: '12%'
            }, {
                data: 'sent_date',
                width: '12%'
            }, {
                data: 'sender',
                width: '12%'
            }, {
                data: 'recipient',
                width: '12%'
            }, {
                data: 'subject',
                width: '27%'
            }, {
                data: 'is_pending',
                width: '8%',
                render: options.convertPendingToString
            }, {
                data: null,
                orderable: false,
                defaultContent: '',
                render: options.emailsTableActions,
                width: '12%'
            }]
        });

        // Add table error handler.
        jse.libs.datatable.error($this, function (event, settings, techNote, message) {
            jse.libs.modal.message({
                title: 'DataTables ' + jse.core.lang.translate('error', 'messages'),
                content: message
            });
        });

        // Add ajax error handler.
        jse.libs.datatable.ajaxComplete($this, function (event, settings, json) {
            if (json.exception === true) {
                jse.core.debug.error('DataTables Processing Error', $this.get(0), json);
                jse.libs.modal.message({
                    title: 'AJAX ' + jse.core.lang.translate('error', 'messages'),
                    content: json.message
                });
            }
        });

        // Combine ".paginator" with the DataTable HTML output in order to create a unique pagination
        // frame at the bottom of the table (executed after table initialization).
        $this.on('init.dt', function (e, settings, json) {
            $('.paginator').appendTo($('#emails-table_wrapper'));
            $('#emails-table_info').appendTo($('.paginator .datatable-components')).css('clear', 'none');
            $('#emails-table_paginate').appendTo($('.paginator .datatable-components')).css('clear', 'none');
        });

        // Recreate the checkbox widgets.
        $this.on('draw.dt', function () {
            $this.find('tbody').attr('data-gx-widget', 'checkbox');
            gx.widgets.init($this); // Initialize the checkbox widget.
        });

        // Add spinner to table loading actions.
        var $spinner;
        $this.on('preXhr.dt', function (e, settings, json) {
            $spinner = jse.libs.loading_spinner.show($this);
        });
        $this.on('xhr.dt', function (e, settings, json) {
            if ($spinner) {
                jse.libs.loading_spinner.hide($spinner);
            }
        });

        // Bind event handlers of the emails table.
        $this.on('click', '#select-all-rows', _onSelectAllRows).on('click', '.send-email', _onSendEmail).on('click', '.forward-email', _onForwardEmail).on('click', '.delete-email', _onDeleteEmail).on('click', '.preview-email', _onPreviewEmail);

        done();
    };

    // Return data to module engine.
    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbImVtYWlscy9lbWFpbHNfdGFibGUuanMiXSwibmFtZXMiOlsiZ3giLCJjb250cm9sbGVycyIsIm1vZHVsZSIsImpzZSIsInNvdXJjZSIsImRhdGEiLCIkdGhpcyIsIiQiLCIkbW9kYWwiLCJkZWZhdWx0cyIsImVtYWlsc1RhYmxlQWN0aW9ucyIsImNvcmUiLCJsYW5nIiwidHJhbnNsYXRlIiwiY29udmVydFBlbmRpbmdUb1N0cmluZyIsInR5cGUiLCJyb3ciLCJtZXRhIiwib3B0aW9ucyIsImV4dGVuZCIsIl9vblNlbGVjdEFsbFJvd3MiLCJldmVudCIsInByb3AiLCJmaW5kIiwiYWRkQ2xhc3MiLCJyZW1vdmVDbGFzcyIsIl9vblNlbmRFbWFpbCIsIiRyb3ciLCJwYXJlbnRzIiwibGlicyIsIm1vZGFsIiwibWVzc2FnZSIsInRpdGxlIiwiY29udGVudCIsImJ1dHRvbnMiLCJ0ZXh0IiwiY2xpY2siLCJkaWFsb2ciLCJlbWFpbCIsImVtYWlscyIsInNlbmRDb2xsZWN0aW9uIiwiZG9uZSIsInJlc3BvbnNlIiwiRGF0YVRhYmxlIiwiYWpheCIsInJlbG9hZCIsImdldEF0dGFjaG1lbnRzU2l6ZSIsImZhaWwiLCJfb25Gb3J3YXJkRW1haWwiLCJyZXNldE1vZGFsIiwibG9hZEVtYWlsT25Nb2RhbCIsInZhbCIsImNsZWFyIiwiZHJhdyIsIndpZHRoIiwiaGVpZ2h0IiwiZGlhbG9nQ2xhc3MiLCJjbG9zZU9uRXNjYXBlIiwiZ2V0RGVmYXVsdE1vZGFsQnV0dG9ucyIsIm9wZW4iLCJjb2xvcml6ZUJ1dHRvbnNGb3JFZGl0TW9kZSIsIl9vbkRlbGV0ZUVtYWlsIiwiZGVsZXRlQ29sbGVjdGlvbiIsIl9vblByZXZpZXdFbWFpbCIsImdldFByZXZpZXdNb2RhbEJ1dHRvbnMiLCJjb2xvcml6ZUJ1dHRvbnNGb3JQcmV2aWV3TW9kZSIsImluaXQiLCJkYXRhdGFibGUiLCJjcmVhdGUiLCJwcm9jZXNzaW5nIiwic2VydmVyU2lkZSIsImRvbSIsImF1dG9XaWR0aCIsImxhbmd1YWdlIiwiY29uZmlnIiwiZ2V0IiwiZ2V0R2VybWFuVHJhbnNsYXRpb24iLCJ1cmwiLCJvcmRlciIsInBhZ2VMZW5ndGgiLCJjb2x1bW5zIiwib3JkZXJhYmxlIiwiZGVmYXVsdENvbnRlbnQiLCJjbGFzc05hbWUiLCJyZW5kZXIiLCJlcnJvciIsInNldHRpbmdzIiwidGVjaE5vdGUiLCJhamF4Q29tcGxldGUiLCJqc29uIiwiZXhjZXB0aW9uIiwiZGVidWciLCJvbiIsImUiLCJhcHBlbmRUbyIsImNzcyIsImF0dHIiLCJ3aWRnZXRzIiwiJHNwaW5uZXIiLCJsb2FkaW5nX3NwaW5uZXIiLCJzaG93IiwiaGlkZSJdLCJtYXBwaW5ncyI6Ijs7QUFBQTs7Ozs7Ozs7OztBQVVBOzs7Ozs7O0FBT0FBLEdBQUdDLFdBQUgsQ0FBZUMsTUFBZixDQUNJLGNBREosRUFHSSxDQUNJQyxJQUFJQyxNQUFKLEdBQWEsOENBRGpCLEVBRUlELElBQUlDLE1BQUosR0FBYSw2Q0FGakIsRUFHSUosR0FBR0ksTUFBSCxHQUFZLGNBSGhCLEVBSUksT0FKSixFQUtJLFdBTEosRUFNSSxXQU5KLENBSEo7O0FBWUk7O0FBRUEsVUFBVUMsSUFBVixFQUFnQjs7QUFFWjs7QUFFQTtBQUNBO0FBQ0E7O0FBRUE7QUFDSTs7Ozs7QUFLQUMsWUFBUUMsRUFBRSxJQUFGLENBTlo7OztBQVFJOzs7OztBQUtBQyxhQUFTRCxFQUFFLGVBQUYsQ0FiYjs7O0FBZUk7Ozs7O0FBS0FFLGVBQVc7QUFDUEMsNEJBQW9CLDhCQUFZO0FBQzVCLG1CQUFPLDhCQUE4Qiw4Q0FBOUIsR0FDRFAsSUFBSVEsSUFBSixDQUFTQyxJQUFULENBQWNDLFNBQWQsQ0FDRSxNQURGLEVBRUUsU0FGRixDQURDLEdBR2MsSUFIZCxHQUdxQixrQ0FIckIsR0FHMEQsU0FIMUQsR0FJSCxpREFKRyxHQUtEVixJQUFJUSxJQUFKLENBQVNDLElBQVQsQ0FBY0MsU0FBZCxDQUF3QixTQUF4QixFQUFtQyxTQUFuQyxDQUxDLEdBTUgsSUFORyxHQU9ILDZCQVBHLEdBTzZCLFNBUDdCLEdBUUgsZ0RBUkcsR0FRZ0RWLElBQUlRLElBQUosQ0FBU0MsSUFBVCxDQUFjQyxTQUFkLENBQy9DLFFBRCtDLEVBQ3JDLFNBRHFDLENBUmhELEdBU3dCLElBVHhCLEdBUytCLCtCQVQvQixHQVNpRSxTQVRqRSxHQVVILGlEQVZHLEdBV0RWLElBQUlRLElBQUosQ0FBU0MsSUFBVCxDQUFjQyxTQUFkLENBQXdCLFNBQXhCLEVBQW1DLFNBQW5DLENBWEMsR0FZSCxJQVpHLEdBYUgsMkJBYkcsR0FhMkIsU0FiM0IsR0FhdUMsUUFiOUM7QUFjSCxTQWhCTTs7QUFrQlBDLGdDQUF3QixnQ0FBVVQsSUFBVixFQUFnQlUsSUFBaEIsRUFBc0JDLEdBQXRCLEVBQTJCQyxJQUEzQixFQUFpQztBQUNyRCxtQkFBUVosU0FDQSxJQURELEdBQ1NGLElBQUlRLElBQUosQ0FBU0MsSUFBVCxDQUFjQyxTQUFkLENBQXdCLGVBQXhCLEVBQXlDLFFBQXpDLENBRFQsR0FDOERWLElBQUlRLElBQUosQ0FBU0MsSUFBVCxDQUFjQyxTQUFkLENBQ2pFLFlBRGlFLEVBQ25ELFFBRG1ELENBRHJFO0FBR0g7QUF0Qk0sS0FwQmY7OztBQTZDSTs7Ozs7QUFLQUssY0FBVVgsRUFBRVksTUFBRixDQUFTLElBQVQsRUFBZSxFQUFmLEVBQW1CVixRQUFuQixFQUE2QkosSUFBN0IsQ0FsRGQ7OztBQW9ESTs7Ozs7QUFLQUgsYUFBUyxFQXpEYjs7QUEyREE7QUFDQTtBQUNBOztBQUVBOzs7OztBQUtBLFFBQUlrQixtQkFBbUIsU0FBbkJBLGdCQUFtQixDQUFVQyxLQUFWLEVBQWlCO0FBQ3BDLFlBQUlkLEVBQUUsSUFBRixFQUFRZSxJQUFSLENBQWEsU0FBYixDQUFKLEVBQTZCO0FBQ3pCaEIsa0JBQU1pQixJQUFOLENBQVcsd0JBQVgsRUFBcUNDLFFBQXJDLENBQThDLFNBQTlDO0FBQ0FsQixrQkFBTWlCLElBQU4sQ0FBVyxzQkFBWCxFQUFtQ0QsSUFBbkMsQ0FBd0MsU0FBeEMsRUFBbUQsSUFBbkQ7QUFDSCxTQUhELE1BR087QUFDSGhCLGtCQUFNaUIsSUFBTixDQUFXLHdCQUFYLEVBQXFDRSxXQUFyQyxDQUFpRCxTQUFqRDtBQUNBbkIsa0JBQU1pQixJQUFOLENBQVcsc0JBQVgsRUFBbUNELElBQW5DLENBQXdDLFNBQXhDLEVBQW1ELEtBQW5EO0FBQ0g7QUFDSixLQVJEOztBQVVBOzs7OztBQUtBLFFBQUlJLGVBQWUsU0FBZkEsWUFBZSxDQUFVTCxLQUFWLEVBQWlCO0FBQ2hDLFlBQUlNLE9BQU9wQixFQUFFLElBQUYsRUFBUXFCLE9BQVIsQ0FBZ0IsSUFBaEIsQ0FBWDs7QUFFQXpCLFlBQUkwQixJQUFKLENBQVNDLEtBQVQsQ0FBZUMsT0FBZixDQUF1QjtBQUNuQkMsbUJBQU83QixJQUFJUSxJQUFKLENBQVNDLElBQVQsQ0FBY0MsU0FBZCxDQUF3QixNQUF4QixFQUFnQyxTQUFoQyxDQURZO0FBRW5Cb0IscUJBQVM5QixJQUFJUSxJQUFKLENBQVNDLElBQVQsQ0FBY0MsU0FBZCxDQUF3QixtQkFBeEIsRUFBNkMsUUFBN0MsQ0FGVTtBQUduQnFCLHFCQUFTLENBQ0w7QUFDSUMsc0JBQU1oQyxJQUFJUSxJQUFKLENBQVNDLElBQVQsQ0FBY0MsU0FBZCxDQUF3QixJQUF4QixFQUE4QixrQkFBOUIsQ0FEVjtBQUVJdUIsdUJBQU8saUJBQVk7QUFDZjdCLHNCQUFFLElBQUYsRUFBUThCLE1BQVIsQ0FBZSxPQUFmO0FBQ0g7QUFKTCxhQURLLEVBT0w7QUFDSUYsc0JBQU1oQyxJQUFJUSxJQUFKLENBQVNDLElBQVQsQ0FBY0MsU0FBZCxDQUF3QixLQUF4QixFQUErQixrQkFBL0IsQ0FEVjtBQUVJdUIsdUJBQU8saUJBQVk7QUFDZix3QkFBSUUsUUFBUVgsS0FBS3RCLElBQUwsRUFBWjtBQUNBRix3QkFBSTBCLElBQUosQ0FBU1UsTUFBVCxDQUFnQkMsY0FBaEIsQ0FBK0IsQ0FBQ0YsS0FBRCxDQUEvQixFQUNLRyxJQURMLENBQ1UsVUFBVUMsUUFBVixFQUFvQjtBQUN0QnBDLDhCQUFNcUMsU0FBTixHQUFrQkMsSUFBbEIsQ0FBdUJDLE1BQXZCO0FBQ0ExQyw0QkFBSTBCLElBQUosQ0FBU1UsTUFBVCxDQUFnQk8sa0JBQWhCLENBQW1DdkMsRUFBRSxtQkFBRixDQUFuQztBQUNILHFCQUpMLEVBS0t3QyxJQUxMLENBS1UsVUFBVUwsUUFBVixFQUFvQjtBQUN0Qiw0QkFBSVYsUUFBUTdCLElBQUlRLElBQUosQ0FBU0MsSUFBVCxDQUFjQyxTQUFkLENBQXdCLE9BQXhCLEVBQWlDLFVBQWpDLENBQVo7O0FBRUFWLDRCQUFJMEIsSUFBSixDQUFTQyxLQUFULENBQWVDLE9BQWYsQ0FBdUI7QUFDbkJDLG1DQUFPQSxLQURZO0FBRW5CQyxxQ0FBU1MsU0FBU1g7QUFGQyx5QkFBdkI7QUFJSCxxQkFaTDtBQWFBeEIsc0JBQUUsSUFBRixFQUFROEIsTUFBUixDQUFlLE9BQWY7QUFDSDtBQWxCTCxhQVBLO0FBSFUsU0FBdkI7QUFnQ0gsS0FuQ0Q7O0FBcUNBOzs7Ozs7O0FBT0EsUUFBSVcsa0JBQWtCLFNBQWxCQSxlQUFrQixDQUFVM0IsS0FBVixFQUFpQjtBQUNuQyxZQUFJaUIsUUFBUS9CLEVBQUUsSUFBRixFQUFRcUIsT0FBUixDQUFnQixJQUFoQixFQUFzQnZCLElBQXRCLEVBQVo7O0FBRUFGLFlBQUkwQixJQUFKLENBQVNVLE1BQVQsQ0FBZ0JVLFVBQWhCLENBQTJCekMsTUFBM0I7QUFDQUwsWUFBSTBCLElBQUosQ0FBU1UsTUFBVCxDQUFnQlcsZ0JBQWhCLENBQWlDWixLQUFqQyxFQUF3QzlCLE1BQXhDOztBQUVBO0FBQ0FBLGVBQU9lLElBQVAsQ0FBWSxXQUFaLEVBQXlCNEIsR0FBekIsQ0FBNkIsRUFBN0I7QUFDQTNDLGVBQU9lLElBQVAsQ0FBWSw2QkFBWixFQUEyQzRCLEdBQTNDLENBQStDLEVBQS9DO0FBQ0EzQyxlQUFPZSxJQUFQLENBQVksaUNBQVosRUFBK0M0QixHQUEvQyxDQUFtRCxFQUFuRDtBQUNBM0MsZUFBT2UsSUFBUCxDQUFZLG1DQUFaLEVBQWlENEIsR0FBakQsQ0FBcUQsRUFBckQ7QUFDQTNDLGVBQU9lLElBQVAsQ0FBWSxpQkFBWixFQUErQm9CLFNBQS9CLEdBQTJDUyxLQUEzQyxHQUFtREMsSUFBbkQ7O0FBRUE3QyxlQUFPNkIsTUFBUCxDQUFjO0FBQ1ZMLG1CQUFPN0IsSUFBSVEsSUFBSixDQUFTQyxJQUFULENBQWNDLFNBQWQsQ0FBd0IsU0FBeEIsRUFBbUMsU0FBbkMsQ0FERztBQUVWeUMsbUJBQU8sSUFGRztBQUdWQyxvQkFBUSxHQUhFO0FBSVZ6QixtQkFBTyxJQUpHO0FBS1YwQix5QkFBYSxjQUxIO0FBTVZDLDJCQUFlLEtBTkw7QUFPVnZCLHFCQUFTL0IsSUFBSTBCLElBQUosQ0FBU1UsTUFBVCxDQUFnQm1CLHNCQUFoQixDQUF1Q2xELE1BQXZDLEVBQStDRixLQUEvQyxDQVBDO0FBUVZxRCxrQkFBTXhELElBQUkwQixJQUFKLENBQVNVLE1BQVQsQ0FBZ0JxQjtBQVJaLFNBQWQ7QUFVSCxLQXZCRDs7QUF5QkE7Ozs7O0FBS0EsUUFBSUMsaUJBQWlCLFNBQWpCQSxjQUFpQixDQUFVeEMsS0FBVixFQUFpQjtBQUNsQyxZQUFJTSxPQUFPcEIsRUFBRSxJQUFGLEVBQVFxQixPQUFSLENBQWdCLElBQWhCLENBQVg7QUFBQSxZQUNJVSxRQUFRWCxLQUFLdEIsSUFBTCxFQURaOztBQUdBRixZQUFJMEIsSUFBSixDQUFTQyxLQUFULENBQWVDLE9BQWYsQ0FBdUI7QUFDbkJDLG1CQUFPN0IsSUFBSVEsSUFBSixDQUFTQyxJQUFULENBQWNDLFNBQWQsQ0FBd0IsUUFBeEIsRUFBa0MsU0FBbEMsQ0FEWTtBQUVuQm9CLHFCQUFTOUIsSUFBSVEsSUFBSixDQUFTQyxJQUFULENBQWNDLFNBQWQsQ0FBd0IscUJBQXhCLEVBQStDLFFBQS9DLENBRlU7QUFHbkJxQixxQkFBUyxDQUNMO0FBQ0lDLHNCQUFNaEMsSUFBSVEsSUFBSixDQUFTQyxJQUFULENBQWNDLFNBQWQsQ0FBd0IsSUFBeEIsRUFBOEIsa0JBQTlCLENBRFY7QUFFSXVCLHVCQUFPLGlCQUFZO0FBQ2Y3QixzQkFBRSxJQUFGLEVBQVE4QixNQUFSLENBQWUsT0FBZjtBQUNIO0FBSkwsYUFESyxFQU9MO0FBQ0lGLHNCQUFNaEMsSUFBSVEsSUFBSixDQUFTQyxJQUFULENBQWNDLFNBQWQsQ0FBd0IsS0FBeEIsRUFBK0Isa0JBQS9CLENBRFY7QUFFSXVCLHVCQUFPLGlCQUFZO0FBQ2ZqQyx3QkFBSTBCLElBQUosQ0FBU1UsTUFBVCxDQUFnQnVCLGdCQUFoQixDQUFpQyxDQUFDeEIsS0FBRCxDQUFqQyxFQUNLRyxJQURMLENBQ1UsVUFBVUMsUUFBVixFQUFvQjtBQUN0QnBDLDhCQUFNcUMsU0FBTixHQUFrQkMsSUFBbEIsQ0FBdUJDLE1BQXZCO0FBQ0ExQyw0QkFBSTBCLElBQUosQ0FBU1UsTUFBVCxDQUFnQk8sa0JBQWhCLENBQW1DdkMsRUFBRSxtQkFBRixDQUFuQztBQUNILHFCQUpMLEVBS0t3QyxJQUxMLENBS1UsVUFBVUwsUUFBVixFQUFvQjtBQUN0Qiw0QkFBSVYsUUFBUTdCLElBQUlRLElBQUosQ0FBU0MsSUFBVCxDQUFjQyxTQUFkLENBQXdCLE9BQXhCLEVBQWlDLFVBQWpDLENBQVo7QUFDQVYsNEJBQUkwQixJQUFKLENBQVNDLEtBQVQsQ0FBZUMsT0FBZixDQUF1QjtBQUNuQkMsbUNBQU9BLEtBRFk7QUFFbkJDLHFDQUFTUyxTQUFTWDtBQUZDLHlCQUF2QjtBQUlILHFCQVhMO0FBWUF4QixzQkFBRSxJQUFGLEVBQVE4QixNQUFSLENBQWUsT0FBZjtBQUNIO0FBaEJMLGFBUEs7QUFIVSxTQUF2QjtBQThCSCxLQWxDRDs7QUFvQ0E7Ozs7Ozs7O0FBUUEsUUFBSTBCLGtCQUFrQixTQUFsQkEsZUFBa0IsQ0FBVTFDLEtBQVYsRUFBaUI7QUFDbkMsWUFBSWlCLFFBQVEvQixFQUFFLElBQUYsRUFBUXFCLE9BQVIsQ0FBZ0IsSUFBaEIsRUFBc0J2QixJQUF0QixFQUFaOztBQUVBRixZQUFJMEIsSUFBSixDQUFTVSxNQUFULENBQWdCVSxVQUFoQixDQUEyQnpDLE1BQTNCO0FBQ0FMLFlBQUkwQixJQUFKLENBQVNVLE1BQVQsQ0FBZ0JXLGdCQUFoQixDQUFpQ1osS0FBakMsRUFBd0M5QixNQUF4Qzs7QUFFQUEsZUFBTzZCLE1BQVAsQ0FBYztBQUNWTCxtQkFBTzdCLElBQUlRLElBQUosQ0FBU0MsSUFBVCxDQUFjQyxTQUFkLENBQXdCLFNBQXhCLEVBQW1DLFNBQW5DLENBREc7QUFFVnlDLG1CQUFPLElBRkc7QUFHVkMsb0JBQVEsR0FIRTtBQUlWekIsbUJBQU8sS0FKRztBQUtWMEIseUJBQWEsY0FMSDtBQU1WQywyQkFBZSxLQU5MO0FBT1Z2QixxQkFBUy9CLElBQUkwQixJQUFKLENBQVNVLE1BQVQsQ0FBZ0J5QixzQkFBaEIsQ0FBdUN4RCxNQUF2QyxFQUErQ0YsS0FBL0MsQ0FQQztBQVFWcUQsa0JBQU14RCxJQUFJMEIsSUFBSixDQUFTVSxNQUFULENBQWdCMEI7QUFSWixTQUFkO0FBVUgsS0FoQkQ7O0FBa0JBO0FBQ0E7QUFDQTs7QUFFQTs7Ozs7Ozs7O0FBU0EvRCxXQUFPZ0UsSUFBUCxHQUFjLFVBQVV6QixJQUFWLEVBQWdCO0FBQzFCO0FBQ0F0QyxZQUFJMEIsSUFBSixDQUFTc0MsU0FBVCxDQUFtQkMsTUFBbkIsQ0FBMEI5RCxLQUExQixFQUFpQztBQUM3QitELHdCQUFZLEtBRGlCO0FBRTdCQyx3QkFBWSxJQUZpQjtBQUc3QkMsaUJBQUssTUFId0I7QUFJN0JDLHVCQUFXLEtBSmtCO0FBSzdCQyxzQkFBV3RFLElBQUlRLElBQUosQ0FBUytELE1BQVQsQ0FBZ0JDLEdBQWhCLENBQW9CLGNBQXBCLE1BQ0gsSUFERSxHQUNNeEUsSUFBSTBCLElBQUosQ0FBU3NDLFNBQVQsQ0FBbUJTLG9CQUFuQixFQUROLEdBQ2tELElBTi9CO0FBTzdCaEMsa0JBQU07QUFDRmlDLHFCQUFLMUUsSUFBSVEsSUFBSixDQUFTK0QsTUFBVCxDQUFnQkMsR0FBaEIsQ0FBb0IsUUFBcEIsSUFBZ0Msc0NBRG5DO0FBRUY1RCxzQkFBTTtBQUZKLGFBUHVCO0FBVzdCK0QsbUJBQU8sQ0FBQyxDQUFDLENBQUQsRUFBSSxNQUFKLENBQUQsQ0FYc0I7QUFZN0JDLHdCQUFZLEVBWmlCO0FBYTdCQyxxQkFBUyxDQUNMO0FBQ0kzRSxzQkFBTSxJQURWO0FBRUk0RSwyQkFBVyxLQUZmO0FBR0lDLGdDQUFnQixnREFIcEI7QUFJSTVCLHVCQUFPLElBSlg7QUFLSTZCLDJCQUFXO0FBTGYsYUFESyxFQVFMO0FBQ0k5RSxzQkFBTSxXQURWO0FBRUk0RSwyQkFBVyxLQUZmO0FBR0kzQix1QkFBTyxJQUhYO0FBSUk2QiwyQkFBVztBQUpmLGFBUkssRUFjTDtBQUNJOUUsc0JBQU0sZUFEVjtBQUVJaUQsdUJBQU87QUFGWCxhQWRLLEVBa0JMO0FBQ0lqRCxzQkFBTSxXQURWO0FBRUlpRCx1QkFBTztBQUZYLGFBbEJLLEVBc0JMO0FBQ0lqRCxzQkFBTSxRQURWO0FBRUlpRCx1QkFBTztBQUZYLGFBdEJLLEVBMEJMO0FBQ0lqRCxzQkFBTSxXQURWO0FBRUlpRCx1QkFBTztBQUZYLGFBMUJLLEVBOEJMO0FBQ0lqRCxzQkFBTSxTQURWO0FBRUlpRCx1QkFBTztBQUZYLGFBOUJLLEVBa0NMO0FBQ0lqRCxzQkFBTSxZQURWO0FBRUlpRCx1QkFBTyxJQUZYO0FBR0k4Qix3QkFBUWxFLFFBQVFKO0FBSHBCLGFBbENLLEVBdUNMO0FBQ0lULHNCQUFNLElBRFY7QUFFSTRFLDJCQUFXLEtBRmY7QUFHSUMsZ0NBQWdCLEVBSHBCO0FBSUlFLHdCQUFRbEUsUUFBUVIsa0JBSnBCO0FBS0k0Qyx1QkFBTztBQUxYLGFBdkNLO0FBYm9CLFNBQWpDOztBQThEQTtBQUNBbkQsWUFBSTBCLElBQUosQ0FBU3NDLFNBQVQsQ0FBbUJrQixLQUFuQixDQUF5Qi9FLEtBQXpCLEVBQWdDLFVBQVVlLEtBQVYsRUFBaUJpRSxRQUFqQixFQUEyQkMsUUFBM0IsRUFBcUN4RCxPQUFyQyxFQUE4QztBQUMxRTVCLGdCQUFJMEIsSUFBSixDQUFTQyxLQUFULENBQWVDLE9BQWYsQ0FBdUI7QUFDbkJDLHVCQUFPLGdCQUFnQjdCLElBQUlRLElBQUosQ0FBU0MsSUFBVCxDQUFjQyxTQUFkLENBQXdCLE9BQXhCLEVBQWlDLFVBQWpDLENBREo7QUFFbkJvQix5QkFBU0Y7QUFGVSxhQUF2QjtBQUlILFNBTEQ7O0FBT0E7QUFDQTVCLFlBQUkwQixJQUFKLENBQVNzQyxTQUFULENBQW1CcUIsWUFBbkIsQ0FBZ0NsRixLQUFoQyxFQUF1QyxVQUFVZSxLQUFWLEVBQWlCaUUsUUFBakIsRUFBMkJHLElBQTNCLEVBQWlDO0FBQ3BFLGdCQUFJQSxLQUFLQyxTQUFMLEtBQW1CLElBQXZCLEVBQTZCO0FBQ3pCdkYsb0JBQUlRLElBQUosQ0FBU2dGLEtBQVQsQ0FBZU4sS0FBZixDQUFxQiw2QkFBckIsRUFBb0QvRSxNQUFNcUUsR0FBTixDQUFVLENBQVYsQ0FBcEQsRUFBa0VjLElBQWxFO0FBQ0F0RixvQkFBSTBCLElBQUosQ0FBU0MsS0FBVCxDQUFlQyxPQUFmLENBQXVCO0FBQ25CQywyQkFBTyxVQUFVN0IsSUFBSVEsSUFBSixDQUFTQyxJQUFULENBQWNDLFNBQWQsQ0FBd0IsT0FBeEIsRUFBaUMsVUFBakMsQ0FERTtBQUVuQm9CLDZCQUFTd0QsS0FBSzFEO0FBRkssaUJBQXZCO0FBSUg7QUFDSixTQVJEOztBQVVBO0FBQ0E7QUFDQXpCLGNBQU1zRixFQUFOLENBQVMsU0FBVCxFQUFvQixVQUFVQyxDQUFWLEVBQWFQLFFBQWIsRUFBdUJHLElBQXZCLEVBQTZCO0FBQzdDbEYsY0FBRSxZQUFGLEVBQWdCdUYsUUFBaEIsQ0FBeUJ2RixFQUFFLHVCQUFGLENBQXpCO0FBQ0FBLGNBQUUsb0JBQUYsRUFDS3VGLFFBREwsQ0FDY3ZGLEVBQUUsa0NBQUYsQ0FEZCxFQUVLd0YsR0FGTCxDQUVTLE9BRlQsRUFFa0IsTUFGbEI7QUFHQXhGLGNBQUUsd0JBQUYsRUFDS3VGLFFBREwsQ0FDY3ZGLEVBQUUsa0NBQUYsQ0FEZCxFQUVLd0YsR0FGTCxDQUVTLE9BRlQsRUFFa0IsTUFGbEI7QUFHSCxTQVJEOztBQVVBO0FBQ0F6RixjQUFNc0YsRUFBTixDQUFTLFNBQVQsRUFBb0IsWUFBWTtBQUM1QnRGLGtCQUFNaUIsSUFBTixDQUFXLE9BQVgsRUFBb0J5RSxJQUFwQixDQUF5QixnQkFBekIsRUFBMkMsVUFBM0M7QUFDQWhHLGVBQUdpRyxPQUFILENBQVcvQixJQUFYLENBQWdCNUQsS0FBaEIsRUFGNEIsQ0FFSjtBQUMzQixTQUhEOztBQUtBO0FBQ0EsWUFBSTRGLFFBQUo7QUFDQTVGLGNBQU1zRixFQUFOLENBQVMsV0FBVCxFQUFzQixVQUFVQyxDQUFWLEVBQWFQLFFBQWIsRUFBdUJHLElBQXZCLEVBQTZCO0FBQy9DUyx1QkFBVy9GLElBQUkwQixJQUFKLENBQVNzRSxlQUFULENBQXlCQyxJQUF6QixDQUE4QjlGLEtBQTlCLENBQVg7QUFDSCxTQUZEO0FBR0FBLGNBQU1zRixFQUFOLENBQVMsUUFBVCxFQUFtQixVQUFVQyxDQUFWLEVBQWFQLFFBQWIsRUFBdUJHLElBQXZCLEVBQTZCO0FBQzVDLGdCQUFJUyxRQUFKLEVBQWM7QUFDVi9GLG9CQUFJMEIsSUFBSixDQUFTc0UsZUFBVCxDQUF5QkUsSUFBekIsQ0FBOEJILFFBQTlCO0FBQ0g7QUFDSixTQUpEOztBQU1BO0FBQ0E1RixjQUNLc0YsRUFETCxDQUNRLE9BRFIsRUFDaUIsa0JBRGpCLEVBQ3FDeEUsZ0JBRHJDLEVBRUt3RSxFQUZMLENBRVEsT0FGUixFQUVpQixhQUZqQixFQUVnQ2xFLFlBRmhDLEVBR0trRSxFQUhMLENBR1EsT0FIUixFQUdpQixnQkFIakIsRUFHbUM1QyxlQUhuQyxFQUlLNEMsRUFKTCxDQUlRLE9BSlIsRUFJaUIsZUFKakIsRUFJa0MvQixjQUpsQyxFQUtLK0IsRUFMTCxDQUtRLE9BTFIsRUFLaUIsZ0JBTGpCLEVBS21DN0IsZUFMbkM7O0FBT0F0QjtBQUNILEtBekhEOztBQTJIQTtBQUNBLFdBQU92QyxNQUFQO0FBQ0gsQ0EzWEwiLCJmaWxlIjoiZW1haWxzL2VtYWlsc190YWJsZS5qcyIsInNvdXJjZXNDb250ZW50IjpbIi8qIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG4gZW1haWxzX3RhYmxlLmpzIDIwMTYtMTAtMTFcbiBHYW1iaW8gR21iSFxuIGh0dHA6Ly93d3cuZ2FtYmlvLmRlXG4gQ29weXJpZ2h0IChjKSAyMDE2IEdhbWJpbyBHbWJIXG4gUmVsZWFzZWQgdW5kZXIgdGhlIEdOVSBHZW5lcmFsIFB1YmxpYyBMaWNlbnNlIChWZXJzaW9uIDIpXG4gW2h0dHA6Ly93d3cuZ251Lm9yZy9saWNlbnNlcy9ncGwtMi4wLmh0bWxdXG4gLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS1cbiAqL1xuXG4vKipcbiAqICMjIEVtYWlscyBUYWJsZSBDb250cm9sbGVyXG4gKlxuICogVGhpcyBjb250cm9sbGVyIHdpbGwgaGFuZGxlIHRoZSBtYWluIHRhYmxlIG9wZXJhdGlvbnMgb2YgdGhlIGFkbWluL2VtYWlscyBwYWdlLlxuICpcbiAqIEBtb2R1bGUgQ29udHJvbGxlcnMvZW1haWxzX3RhYmxlXG4gKi9cbmd4LmNvbnRyb2xsZXJzLm1vZHVsZShcbiAgICAnZW1haWxzX3RhYmxlJyxcblxuICAgIFtcbiAgICAgICAganNlLnNvdXJjZSArICcvdmVuZG9yL2RhdGF0YWJsZXMvanF1ZXJ5LmRhdGFUYWJsZXMubWluLmNzcycsXG4gICAgICAgIGpzZS5zb3VyY2UgKyAnL3ZlbmRvci9kYXRhdGFibGVzL2pxdWVyeS5kYXRhVGFibGVzLm1pbi5qcycsXG4gICAgICAgIGd4LnNvdXJjZSArICcvbGlicy9lbWFpbHMnLFxuICAgICAgICAnbW9kYWwnLFxuICAgICAgICAnZGF0YXRhYmxlJyxcbiAgICAgICAgJ25vcm1hbGl6ZSdcbiAgICBdLFxuXG4gICAgLyoqIEBsZW5kcyBtb2R1bGU6Q29udHJvbGxlcnMvZW1haWxzX3RhYmxlICovXG5cbiAgICBmdW5jdGlvbiAoZGF0YSkge1xuXG4gICAgICAgICd1c2Ugc3RyaWN0JztcblxuICAgICAgICAvLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS1cbiAgICAgICAgLy8gVkFSSUFCTEUgREVGSU5JVElPTlxuICAgICAgICAvLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS1cblxuICAgICAgICB2YXJcbiAgICAgICAgICAgIC8qKlxuICAgICAgICAgICAgICogTW9kdWxlIFJlZmVyZW5jZVxuICAgICAgICAgICAgICpcbiAgICAgICAgICAgICAqIEB0eXBlIHtvYmplY3R9XG4gICAgICAgICAgICAgKi9cbiAgICAgICAgICAgICR0aGlzID0gJCh0aGlzKSxcblxuICAgICAgICAgICAgLyoqXG4gICAgICAgICAgICAgKiBNb2RhbCBTZWxlY3RvclxuICAgICAgICAgICAgICpcbiAgICAgICAgICAgICAqIEB0eXBlIHtvYmplY3R9XG4gICAgICAgICAgICAgKi9cbiAgICAgICAgICAgICRtb2RhbCA9ICQoJyNlbWFpbHMtbW9kYWwnKSxcblxuICAgICAgICAgICAgLyoqXG4gICAgICAgICAgICAgKiBEZWZhdWx0IE1vZHVsZSBPcHRpb25zXG4gICAgICAgICAgICAgKlxuICAgICAgICAgICAgICogQHR5cGUge29iamVjdH1cbiAgICAgICAgICAgICAqL1xuICAgICAgICAgICAgZGVmYXVsdHMgPSB7XG4gICAgICAgICAgICAgICAgZW1haWxzVGFibGVBY3Rpb25zOiBmdW5jdGlvbiAoKSB7XG4gICAgICAgICAgICAgICAgICAgIHJldHVybiAnPGRpdiBjbGFzcz1cInJvdy1hY3Rpb25zXCI+JyArICc8c3BhbiBjbGFzcz1cInNlbmQtZW1haWwgYWN0aW9uLWl0ZW1cIiB0aXRsZT1cIidcbiAgICAgICAgICAgICAgICAgICAgICAgICsganNlLmNvcmUubGFuZy50cmFuc2xhdGUoXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgJ3NlbmQnLFxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICdidXR0b25zJykgKyAnXCI+JyArICc8aSBjbGFzcz1cImZhIGZhLWVudmVsb3BlLW9cIj48L2k+JyArICc8L3NwYW4+JyArXG4gICAgICAgICAgICAgICAgICAgICAgICAnPHNwYW4gY2xhc3M9XCJmb3J3YXJkLWVtYWlsIGFjdGlvbi1pdGVtXCIgdGl0bGU9XCInXG4gICAgICAgICAgICAgICAgICAgICAgICArIGpzZS5jb3JlLmxhbmcudHJhbnNsYXRlKCdmb3J3YXJkJywgJ2J1dHRvbnMnKSArXG4gICAgICAgICAgICAgICAgICAgICAgICAnXCI+JyArXG4gICAgICAgICAgICAgICAgICAgICAgICAnPGkgY2xhc3M9XCJmYSBmYS1zaGFyZVwiPjwvaT4nICsgJzwvc3Bhbj4nICtcbiAgICAgICAgICAgICAgICAgICAgICAgICc8c3BhbiBjbGFzcz1cImRlbGV0ZS1lbWFpbCBhY3Rpb24taXRlbVwiIHRpdGxlPVwiJyArIGpzZS5jb3JlLmxhbmcudHJhbnNsYXRlKFxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICdkZWxldGUnLCAnYnV0dG9ucycpICsgJ1wiPicgKyAnPGkgY2xhc3M9XCJmYSBmYS10cmFzaC1vXCI+PC9pPicgKyAnPC9zcGFuPicgK1xuICAgICAgICAgICAgICAgICAgICAgICAgJzxzcGFuIGNsYXNzPVwicHJldmlldy1lbWFpbCBhY3Rpb24taXRlbVwiIHRpdGxlPVwiJ1xuICAgICAgICAgICAgICAgICAgICAgICAgKyBqc2UuY29yZS5sYW5nLnRyYW5zbGF0ZSgncHJldmlldycsICdidXR0b25zJykgK1xuICAgICAgICAgICAgICAgICAgICAgICAgJ1wiPicgK1xuICAgICAgICAgICAgICAgICAgICAgICAgJzxpIGNsYXNzPVwiZmEgZmEtZXllXCI+PC9pPicgKyAnPC9zcGFuPicgKyAnPC9kaXY+JztcbiAgICAgICAgICAgICAgICB9LFxuXG4gICAgICAgICAgICAgICAgY29udmVydFBlbmRpbmdUb1N0cmluZzogZnVuY3Rpb24gKGRhdGEsIHR5cGUsIHJvdywgbWV0YSkge1xuICAgICAgICAgICAgICAgICAgICByZXR1cm4gKGRhdGFcbiAgICAgICAgICAgICAgICAgICAgICAgID09PSB0cnVlKSA/IGpzZS5jb3JlLmxhbmcudHJhbnNsYXRlKCdlbWFpbF9wZW5kaW5nJywgJ2VtYWlscycpIDoganNlLmNvcmUubGFuZy50cmFuc2xhdGUoXG4gICAgICAgICAgICAgICAgICAgICAgICAnZW1haWxfc2VudCcsICdlbWFpbHMnKTtcbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICB9LFxuXG4gICAgICAgICAgICAvKipcbiAgICAgICAgICAgICAqIEZpbmFsIE1vZHVsZSBPcHRpb25zXG4gICAgICAgICAgICAgKlxuICAgICAgICAgICAgICogQHR5cGUge29iamVjdH1cbiAgICAgICAgICAgICAqL1xuICAgICAgICAgICAgb3B0aW9ucyA9ICQuZXh0ZW5kKHRydWUsIHt9LCBkZWZhdWx0cywgZGF0YSksXG5cbiAgICAgICAgICAgIC8qKlxuICAgICAgICAgICAgICogTW9kdWxlIE9iamVjdFxuICAgICAgICAgICAgICpcbiAgICAgICAgICAgICAqIEB0eXBlIHtvYmplY3R9XG4gICAgICAgICAgICAgKi9cbiAgICAgICAgICAgIG1vZHVsZSA9IHt9O1xuXG4gICAgICAgIC8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxuICAgICAgICAvLyBFVkVOVCBIQU5ETEVSU1xuICAgICAgICAvLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS1cblxuICAgICAgICAvKipcbiAgICAgICAgICogVG9nZ2xlIHJvdyBzZWxlY3Rpb24gZm9yIG1haW4gcGFnZSB0YWJsZS5cbiAgICAgICAgICpcbiAgICAgICAgICogQHBhcmFtIHtvYmplY3R9IGV2ZW50IENvbnRhaW5zIGV2ZW50IGluZm9ybWF0aW9uLlxuICAgICAgICAgKi9cbiAgICAgICAgdmFyIF9vblNlbGVjdEFsbFJvd3MgPSBmdW5jdGlvbiAoZXZlbnQpIHtcbiAgICAgICAgICAgIGlmICgkKHRoaXMpLnByb3AoJ2NoZWNrZWQnKSkge1xuICAgICAgICAgICAgICAgICR0aGlzLmZpbmQoJ3Rib2R5IC5zaW5nbGUtY2hlY2tib3gnKS5hZGRDbGFzcygnY2hlY2tlZCcpO1xuICAgICAgICAgICAgICAgICR0aGlzLmZpbmQoJ3Rib2R5IGlucHV0OmNoZWNrYm94JykucHJvcCgnY2hlY2tlZCcsIHRydWUpO1xuICAgICAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAgICAgICAkdGhpcy5maW5kKCd0Ym9keSAuc2luZ2xlLWNoZWNrYm94JykucmVtb3ZlQ2xhc3MoJ2NoZWNrZWQnKTtcbiAgICAgICAgICAgICAgICAkdGhpcy5maW5kKCd0Ym9keSBpbnB1dDpjaGVja2JveCcpLnByb3AoJ2NoZWNrZWQnLCBmYWxzZSk7XG4gICAgICAgICAgICB9XG4gICAgICAgIH07XG5cbiAgICAgICAgLyoqXG4gICAgICAgICAqIFdpbGwgc2VuZCB0aGUgZW1haWwgdG8gaXRzIGNvbnRhY3RzIChldmVuIGlmIGl0cyBzdGF0dXMgaXMgXCJzZW50XCIpLlxuICAgICAgICAgKlxuICAgICAgICAgKiBAcGFyYW0ge29iamVjdH0gZXZlbnQgQ29udGFpbnMgZXZlbnQgaW5mb3JtYXRpb24uXG4gICAgICAgICAqL1xuICAgICAgICB2YXIgX29uU2VuZEVtYWlsID0gZnVuY3Rpb24gKGV2ZW50KSB7XG4gICAgICAgICAgICB2YXIgJHJvdyA9ICQodGhpcykucGFyZW50cygndHInKTtcblxuICAgICAgICAgICAganNlLmxpYnMubW9kYWwubWVzc2FnZSh7XG4gICAgICAgICAgICAgICAgdGl0bGU6IGpzZS5jb3JlLmxhbmcudHJhbnNsYXRlKCdzZW5kJywgJ2J1dHRvbnMnKSxcbiAgICAgICAgICAgICAgICBjb250ZW50OiBqc2UuY29yZS5sYW5nLnRyYW5zbGF0ZSgncHJvbXB0X3NlbmRfZW1haWwnLCAnZW1haWxzJyksXG4gICAgICAgICAgICAgICAgYnV0dG9uczogW1xuICAgICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICAgICB0ZXh0OiBqc2UuY29yZS5sYW5nLnRyYW5zbGF0ZSgnbm8nLCAnbGlnaHRib3hfYnV0dG9ucycpLFxuICAgICAgICAgICAgICAgICAgICAgICAgY2xpY2s6IGZ1bmN0aW9uICgpIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAkKHRoaXMpLmRpYWxvZygnY2xvc2UnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAgICAgdGV4dDoganNlLmNvcmUubGFuZy50cmFuc2xhdGUoJ3llcycsICdsaWdodGJveF9idXR0b25zJyksXG4gICAgICAgICAgICAgICAgICAgICAgICBjbGljazogZnVuY3Rpb24gKCkge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIHZhciBlbWFpbCA9ICRyb3cuZGF0YSgpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIGpzZS5saWJzLmVtYWlscy5zZW5kQ29sbGVjdGlvbihbZW1haWxdKVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAuZG9uZShmdW5jdGlvbiAocmVzcG9uc2UpIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICR0aGlzLkRhdGFUYWJsZSgpLmFqYXgucmVsb2FkKCk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBqc2UubGlicy5lbWFpbHMuZ2V0QXR0YWNobWVudHNTaXplKCQoJyNhdHRhY2htZW50cy1zaXplJykpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICB9KVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAuZmFpbChmdW5jdGlvbiAocmVzcG9uc2UpIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIHZhciB0aXRsZSA9IGpzZS5jb3JlLmxhbmcudHJhbnNsYXRlKCdlcnJvcicsICdtZXNzYWdlcycpO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBqc2UubGlicy5tb2RhbC5tZXNzYWdlKHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICB0aXRsZTogdGl0bGUsXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgY29udGVudDogcmVzcG9uc2UubWVzc2FnZVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgfSk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIH0pO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICQodGhpcykuZGlhbG9nKCdjbG9zZScpO1xuICAgICAgICAgICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgfSk7XG4gICAgICAgIH07XG5cbiAgICAgICAgLyoqXG4gICAgICAgICAqIERpc3BsYXkgbW9kYWwgd2l0aCBlbWFpbCBpbmZvcm1hdGlvbiBidXQgd2l0aG91dCBjb250YWN0cy5cbiAgICAgICAgICpcbiAgICAgICAgICogVGhlIHVzZXIgd2lsbCBiZSBhYmxlIHRvIHNldCBuZXcgY29udGFjdHMgYW5kIHNlbmQgdGhlIGVtYWlsIChraW5kIG9mIFwiZHVwbGljYXRlXCIgbWV0aG9kKS5cbiAgICAgICAgICpcbiAgICAgICAgICogQHBhcmFtIHtvYmplY3R9IGV2ZW50IENvbnRhaW5zIGV2ZW50IGluZm9ybWF0aW9uLlxuICAgICAgICAgKi9cbiAgICAgICAgdmFyIF9vbkZvcndhcmRFbWFpbCA9IGZ1bmN0aW9uIChldmVudCkge1xuICAgICAgICAgICAgdmFyIGVtYWlsID0gJCh0aGlzKS5wYXJlbnRzKCd0cicpLmRhdGEoKTtcblxuICAgICAgICAgICAganNlLmxpYnMuZW1haWxzLnJlc2V0TW9kYWwoJG1vZGFsKTtcbiAgICAgICAgICAgIGpzZS5saWJzLmVtYWlscy5sb2FkRW1haWxPbk1vZGFsKGVtYWlsLCAkbW9kYWwpO1xuXG4gICAgICAgICAgICAvLyBDbGVhciBjb250YWN0IGZpZWxkcyBidXQgbGV0IHRoZSByZXN0IG9mIHRoZSBkYXRhIHVudG91Y2hlZC5cbiAgICAgICAgICAgICRtb2RhbC5maW5kKCcjZW1haWwtaWQnKS52YWwoJycpO1xuICAgICAgICAgICAgJG1vZGFsLmZpbmQoJyNzZW5kZXItZW1haWwsICNzZW5kZXItbmFtZScpLnZhbCgnJyk7XG4gICAgICAgICAgICAkbW9kYWwuZmluZCgnI3JlcGx5LXRvLWVtYWlsLCAjcmVwbHktdG8tbmFtZScpLnZhbCgnJyk7XG4gICAgICAgICAgICAkbW9kYWwuZmluZCgnI3JlY2lwaWVudC1lbWFpbCwgI3JlY2lwaWVudC1uYW1lJykudmFsKCcnKTtcbiAgICAgICAgICAgICRtb2RhbC5maW5kKCcjY29udGFjdHMtdGFibGUnKS5EYXRhVGFibGUoKS5jbGVhcigpLmRyYXcoKTtcblxuICAgICAgICAgICAgJG1vZGFsLmRpYWxvZyh7XG4gICAgICAgICAgICAgICAgdGl0bGU6IGpzZS5jb3JlLmxhbmcudHJhbnNsYXRlKCdmb3J3YXJkJywgJ2J1dHRvbnMnKSxcbiAgICAgICAgICAgICAgICB3aWR0aDogMTAwMCxcbiAgICAgICAgICAgICAgICBoZWlnaHQ6IDgwMCxcbiAgICAgICAgICAgICAgICBtb2RhbDogdHJ1ZSxcbiAgICAgICAgICAgICAgICBkaWFsb2dDbGFzczogJ2d4LWNvbnRhaW5lcicsXG4gICAgICAgICAgICAgICAgY2xvc2VPbkVzY2FwZTogZmFsc2UsXG4gICAgICAgICAgICAgICAgYnV0dG9uczoganNlLmxpYnMuZW1haWxzLmdldERlZmF1bHRNb2RhbEJ1dHRvbnMoJG1vZGFsLCAkdGhpcyksXG4gICAgICAgICAgICAgICAgb3BlbjoganNlLmxpYnMuZW1haWxzLmNvbG9yaXplQnV0dG9uc0ZvckVkaXRNb2RlXG4gICAgICAgICAgICB9KTtcbiAgICAgICAgfTtcblxuICAgICAgICAvKipcbiAgICAgICAgICogRGVsZXRlIHNlbGVjdGVkIHJvdyBlbWFpbC5cbiAgICAgICAgICpcbiAgICAgICAgICogQHBhcmFtIHtvYmplY3R9IGV2ZW50IENvbnRhaW5zIGV2ZW50IGluZm9ybWF0aW9uLlxuICAgICAgICAgKi9cbiAgICAgICAgdmFyIF9vbkRlbGV0ZUVtYWlsID0gZnVuY3Rpb24gKGV2ZW50KSB7XG4gICAgICAgICAgICB2YXIgJHJvdyA9ICQodGhpcykucGFyZW50cygndHInKSxcbiAgICAgICAgICAgICAgICBlbWFpbCA9ICRyb3cuZGF0YSgpO1xuXG4gICAgICAgICAgICBqc2UubGlicy5tb2RhbC5tZXNzYWdlKHtcbiAgICAgICAgICAgICAgICB0aXRsZToganNlLmNvcmUubGFuZy50cmFuc2xhdGUoJ2RlbGV0ZScsICdidXR0b25zJyksXG4gICAgICAgICAgICAgICAgY29udGVudDoganNlLmNvcmUubGFuZy50cmFuc2xhdGUoJ3Byb21wdF9kZWxldGVfZW1haWwnLCAnZW1haWxzJyksXG4gICAgICAgICAgICAgICAgYnV0dG9uczogW1xuICAgICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICAgICB0ZXh0OiBqc2UuY29yZS5sYW5nLnRyYW5zbGF0ZSgnbm8nLCAnbGlnaHRib3hfYnV0dG9ucycpLFxuICAgICAgICAgICAgICAgICAgICAgICAgY2xpY2s6IGZ1bmN0aW9uICgpIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAkKHRoaXMpLmRpYWxvZygnY2xvc2UnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAgICAgdGV4dDoganNlLmNvcmUubGFuZy50cmFuc2xhdGUoJ3llcycsICdsaWdodGJveF9idXR0b25zJyksXG4gICAgICAgICAgICAgICAgICAgICAgICBjbGljazogZnVuY3Rpb24gKCkge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIGpzZS5saWJzLmVtYWlscy5kZWxldGVDb2xsZWN0aW9uKFtlbWFpbF0pXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIC5kb25lKGZ1bmN0aW9uIChyZXNwb25zZSkge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgJHRoaXMuRGF0YVRhYmxlKCkuYWpheC5yZWxvYWQoKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIGpzZS5saWJzLmVtYWlscy5nZXRBdHRhY2htZW50c1NpemUoJCgnI2F0dGFjaG1lbnRzLXNpemUnKSk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIH0pXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIC5mYWlsKGZ1bmN0aW9uIChyZXNwb25zZSkge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgdmFyIHRpdGxlID0ganNlLmNvcmUubGFuZy50cmFuc2xhdGUoJ2Vycm9yJywgJ21lc3NhZ2VzJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBqc2UubGlicy5tb2RhbC5tZXNzYWdlKHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICB0aXRsZTogdGl0bGUsXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgY29udGVudDogcmVzcG9uc2UubWVzc2FnZVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgfSk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIH0pO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICQodGhpcykuZGlhbG9nKCdjbG9zZScpO1xuICAgICAgICAgICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgfSk7XG4gICAgICAgIH07XG5cbiAgICAgICAgLyoqXG4gICAgICAgICAqIERpc3BsYXkgbW9kYWwgd2l0aCBlbWFpbCBpbmZvcm1hdGlvblxuICAgICAgICAgKlxuICAgICAgICAgKiBUaGUgdXNlciBjYW4gc2VsZWN0IGFuIGFjdGlvbiB0byBwZXJmb3JtIHVwb24gdGhlIHByZXZpZXdlZCBlbWFpbCAoU2VuZCwgRm9yd2FyZCxcbiAgICAgICAgICogRGVsZXRlLCBDbG9zZSkuXG4gICAgICAgICAqXG4gICAgICAgICAqIEBwYXJhbSAge29iamVjdH0gZXZlbnQgQ29udGFpbnMgZXZlbnQgaW5mb3JtYXRpb24uXG4gICAgICAgICAqL1xuICAgICAgICB2YXIgX29uUHJldmlld0VtYWlsID0gZnVuY3Rpb24gKGV2ZW50KSB7XG4gICAgICAgICAgICB2YXIgZW1haWwgPSAkKHRoaXMpLnBhcmVudHMoJ3RyJykuZGF0YSgpO1xuXG4gICAgICAgICAgICBqc2UubGlicy5lbWFpbHMucmVzZXRNb2RhbCgkbW9kYWwpO1xuICAgICAgICAgICAganNlLmxpYnMuZW1haWxzLmxvYWRFbWFpbE9uTW9kYWwoZW1haWwsICRtb2RhbCk7XG5cbiAgICAgICAgICAgICRtb2RhbC5kaWFsb2coe1xuICAgICAgICAgICAgICAgIHRpdGxlOiBqc2UuY29yZS5sYW5nLnRyYW5zbGF0ZSgncHJldmlldycsICdidXR0b25zJyksXG4gICAgICAgICAgICAgICAgd2lkdGg6IDEwMDAsXG4gICAgICAgICAgICAgICAgaGVpZ2h0OiA4MDAsXG4gICAgICAgICAgICAgICAgbW9kYWw6IGZhbHNlLFxuICAgICAgICAgICAgICAgIGRpYWxvZ0NsYXNzOiAnZ3gtY29udGFpbmVyJyxcbiAgICAgICAgICAgICAgICBjbG9zZU9uRXNjYXBlOiBmYWxzZSxcbiAgICAgICAgICAgICAgICBidXR0b25zOiBqc2UubGlicy5lbWFpbHMuZ2V0UHJldmlld01vZGFsQnV0dG9ucygkbW9kYWwsICR0aGlzKSxcbiAgICAgICAgICAgICAgICBvcGVuOiBqc2UubGlicy5lbWFpbHMuY29sb3JpemVCdXR0b25zRm9yUHJldmlld01vZGVcbiAgICAgICAgICAgIH0pO1xuICAgICAgICB9O1xuXG4gICAgICAgIC8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxuICAgICAgICAvLyBJTklUSUFMSVpBVElPTlxuICAgICAgICAvLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS1cblxuICAgICAgICAvKipcbiAgICAgICAgICogSW5pdGlhbGl6ZSBtZXRob2Qgb2YgdGhlIG1vZHVsZSwgY2FsbGVkIGJ5IHRoZSBlbmdpbmUuXG4gICAgICAgICAqXG4gICAgICAgICAqIFRoZSBlbWFpbHMgdGFibGUgb3BlcmF0ZXMgd2l0aCBzZXJ2ZXIgcHJvY2Vzc2luZyBiZWNhdXNlIGl0IGlzIG11Y2ggZmFzdGVyIGFuZCBlZmZpY2llbnQgdGhhbiBwcmVwYXJpbmdcbiAgICAgICAgICogYW5kIHNlbmRpbmcgYWxsIHRoZSByZWNvcmRzIGluIGV2ZXJ5IEFKQVggcmVxdWVzdC4gQ2hlY2sgdGhlIEVtYWlscy9EYXRhVGFibGUgY29udHJvbGxlciBtZXRob2QgZm9yXG4gICAgICAgICAqIHJlcXVlc3RlZCBkYXRhIGFuZCB0aGUgZm9sbG93aW5nIGxpbmsgZm9yIG1vcmUgaW5mbyBhYm91dCBzZXJ2ZXIgcHJvY2Vzc2luZyBpbiBEYXRhVGFibGVzLlxuICAgICAgICAgKlxuICAgICAgICAgKiB7QGxpbmsgaHR0cDovL3d3dy5kYXRhdGFibGVzLm5ldC9tYW51YWwvc2VydmVyLXNpZGV9XG4gICAgICAgICAqL1xuICAgICAgICBtb2R1bGUuaW5pdCA9IGZ1bmN0aW9uIChkb25lKSB7XG4gICAgICAgICAgICAvLyBDcmVhdGUgYSBEYXRhVGFibGUgaW5zdGFuY2UgZm9yIHRoZSBlbWFpbCByZWNvcmRzLlxuICAgICAgICAgICAganNlLmxpYnMuZGF0YXRhYmxlLmNyZWF0ZSgkdGhpcywge1xuICAgICAgICAgICAgICAgIHByb2Nlc3Npbmc6IGZhbHNlLFxuICAgICAgICAgICAgICAgIHNlcnZlclNpZGU6IHRydWUsXG4gICAgICAgICAgICAgICAgZG9tOiAncnRpcCcsXG4gICAgICAgICAgICAgICAgYXV0b1dpZHRoOiBmYWxzZSxcbiAgICAgICAgICAgICAgICBsYW5ndWFnZTogKGpzZS5jb3JlLmNvbmZpZy5nZXQoJ2xhbmd1YWdlQ29kZScpXG4gICAgICAgICAgICAgICAgICAgID09PSAnZGUnKSA/IGpzZS5saWJzLmRhdGF0YWJsZS5nZXRHZXJtYW5UcmFuc2xhdGlvbigpIDogbnVsbCxcbiAgICAgICAgICAgICAgICBhamF4OiB7XG4gICAgICAgICAgICAgICAgICAgIHVybDoganNlLmNvcmUuY29uZmlnLmdldCgnYXBwVXJsJykgKyAnL2FkbWluL2FkbWluLnBocD9kbz1FbWFpbHMvRGF0YVRhYmxlJyxcbiAgICAgICAgICAgICAgICAgICAgdHlwZTogJ1BPU1QnXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICBvcmRlcjogW1syLCAnZGVzYyddXSxcbiAgICAgICAgICAgICAgICBwYWdlTGVuZ3RoOiAyMCxcbiAgICAgICAgICAgICAgICBjb2x1bW5zOiBbXG4gICAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgICAgIGRhdGE6IG51bGwsXG4gICAgICAgICAgICAgICAgICAgICAgICBvcmRlcmFibGU6IGZhbHNlLFxuICAgICAgICAgICAgICAgICAgICAgICAgZGVmYXVsdENvbnRlbnQ6ICc8aW5wdXQgdHlwZT1cImNoZWNrYm94XCIgZGF0YS1zaW5nbGVfY2hlY2tib3ggLz4nLFxuICAgICAgICAgICAgICAgICAgICAgICAgd2lkdGg6ICcyJScsXG4gICAgICAgICAgICAgICAgICAgICAgICBjbGFzc05hbWU6ICdkdC1oZWFkLWNlbnRlciBkdC1ib2R5LWNlbnRlcidcbiAgICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAgICAgZGF0YTogJ3Jvd19jb3VudCcsXG4gICAgICAgICAgICAgICAgICAgICAgICBvcmRlcmFibGU6IGZhbHNlLFxuICAgICAgICAgICAgICAgICAgICAgICAgd2lkdGg6ICczJScsXG4gICAgICAgICAgICAgICAgICAgICAgICBjbGFzc05hbWU6ICdkdC1oZWFkLWNlbnRlciBkdC1ib2R5LWNlbnRlcidcbiAgICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAgICAgZGF0YTogJ2NyZWF0aW9uX2RhdGUnLFxuICAgICAgICAgICAgICAgICAgICAgICAgd2lkdGg6ICcxMiUnXG4gICAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgICAgIGRhdGE6ICdzZW50X2RhdGUnLFxuICAgICAgICAgICAgICAgICAgICAgICAgd2lkdGg6ICcxMiUnXG4gICAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgICAgIGRhdGE6ICdzZW5kZXInLFxuICAgICAgICAgICAgICAgICAgICAgICAgd2lkdGg6ICcxMiUnXG4gICAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgICAgIGRhdGE6ICdyZWNpcGllbnQnLFxuICAgICAgICAgICAgICAgICAgICAgICAgd2lkdGg6ICcxMiUnXG4gICAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgICAgIGRhdGE6ICdzdWJqZWN0JyxcbiAgICAgICAgICAgICAgICAgICAgICAgIHdpZHRoOiAnMjclJ1xuICAgICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICAgICBkYXRhOiAnaXNfcGVuZGluZycsXG4gICAgICAgICAgICAgICAgICAgICAgICB3aWR0aDogJzglJyxcbiAgICAgICAgICAgICAgICAgICAgICAgIHJlbmRlcjogb3B0aW9ucy5jb252ZXJ0UGVuZGluZ1RvU3RyaW5nXG4gICAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgICAgIGRhdGE6IG51bGwsXG4gICAgICAgICAgICAgICAgICAgICAgICBvcmRlcmFibGU6IGZhbHNlLFxuICAgICAgICAgICAgICAgICAgICAgICAgZGVmYXVsdENvbnRlbnQ6ICcnLFxuICAgICAgICAgICAgICAgICAgICAgICAgcmVuZGVyOiBvcHRpb25zLmVtYWlsc1RhYmxlQWN0aW9ucyxcbiAgICAgICAgICAgICAgICAgICAgICAgIHdpZHRoOiAnMTIlJ1xuICAgICAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgfSk7XG5cbiAgICAgICAgICAgIC8vIEFkZCB0YWJsZSBlcnJvciBoYW5kbGVyLlxuICAgICAgICAgICAganNlLmxpYnMuZGF0YXRhYmxlLmVycm9yKCR0aGlzLCBmdW5jdGlvbiAoZXZlbnQsIHNldHRpbmdzLCB0ZWNoTm90ZSwgbWVzc2FnZSkge1xuICAgICAgICAgICAgICAgIGpzZS5saWJzLm1vZGFsLm1lc3NhZ2Uoe1xuICAgICAgICAgICAgICAgICAgICB0aXRsZTogJ0RhdGFUYWJsZXMgJyArIGpzZS5jb3JlLmxhbmcudHJhbnNsYXRlKCdlcnJvcicsICdtZXNzYWdlcycpLFxuICAgICAgICAgICAgICAgICAgICBjb250ZW50OiBtZXNzYWdlXG4gICAgICAgICAgICAgICAgfSk7XG4gICAgICAgICAgICB9KTtcblxuICAgICAgICAgICAgLy8gQWRkIGFqYXggZXJyb3IgaGFuZGxlci5cbiAgICAgICAgICAgIGpzZS5saWJzLmRhdGF0YWJsZS5hamF4Q29tcGxldGUoJHRoaXMsIGZ1bmN0aW9uIChldmVudCwgc2V0dGluZ3MsIGpzb24pIHtcbiAgICAgICAgICAgICAgICBpZiAoanNvbi5leGNlcHRpb24gPT09IHRydWUpIHtcbiAgICAgICAgICAgICAgICAgICAganNlLmNvcmUuZGVidWcuZXJyb3IoJ0RhdGFUYWJsZXMgUHJvY2Vzc2luZyBFcnJvcicsICR0aGlzLmdldCgwKSwganNvbik7XG4gICAgICAgICAgICAgICAgICAgIGpzZS5saWJzLm1vZGFsLm1lc3NhZ2Uoe1xuICAgICAgICAgICAgICAgICAgICAgICAgdGl0bGU6ICdBSkFYICcgKyBqc2UuY29yZS5sYW5nLnRyYW5zbGF0ZSgnZXJyb3InLCAnbWVzc2FnZXMnKSxcbiAgICAgICAgICAgICAgICAgICAgICAgIGNvbnRlbnQ6IGpzb24ubWVzc2FnZVxuICAgICAgICAgICAgICAgICAgICB9KTtcbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICB9KTtcblxuICAgICAgICAgICAgLy8gQ29tYmluZSBcIi5wYWdpbmF0b3JcIiB3aXRoIHRoZSBEYXRhVGFibGUgSFRNTCBvdXRwdXQgaW4gb3JkZXIgdG8gY3JlYXRlIGEgdW5pcXVlIHBhZ2luYXRpb25cbiAgICAgICAgICAgIC8vIGZyYW1lIGF0IHRoZSBib3R0b20gb2YgdGhlIHRhYmxlIChleGVjdXRlZCBhZnRlciB0YWJsZSBpbml0aWFsaXphdGlvbikuXG4gICAgICAgICAgICAkdGhpcy5vbignaW5pdC5kdCcsIGZ1bmN0aW9uIChlLCBzZXR0aW5ncywganNvbikge1xuICAgICAgICAgICAgICAgICQoJy5wYWdpbmF0b3InKS5hcHBlbmRUbygkKCcjZW1haWxzLXRhYmxlX3dyYXBwZXInKSk7XG4gICAgICAgICAgICAgICAgJCgnI2VtYWlscy10YWJsZV9pbmZvJylcbiAgICAgICAgICAgICAgICAgICAgLmFwcGVuZFRvKCQoJy5wYWdpbmF0b3IgLmRhdGF0YWJsZS1jb21wb25lbnRzJykpXG4gICAgICAgICAgICAgICAgICAgIC5jc3MoJ2NsZWFyJywgJ25vbmUnKTtcbiAgICAgICAgICAgICAgICAkKCcjZW1haWxzLXRhYmxlX3BhZ2luYXRlJylcbiAgICAgICAgICAgICAgICAgICAgLmFwcGVuZFRvKCQoJy5wYWdpbmF0b3IgLmRhdGF0YWJsZS1jb21wb25lbnRzJykpXG4gICAgICAgICAgICAgICAgICAgIC5jc3MoJ2NsZWFyJywgJ25vbmUnKTtcbiAgICAgICAgICAgIH0pO1xuXG4gICAgICAgICAgICAvLyBSZWNyZWF0ZSB0aGUgY2hlY2tib3ggd2lkZ2V0cy5cbiAgICAgICAgICAgICR0aGlzLm9uKCdkcmF3LmR0JywgZnVuY3Rpb24gKCkge1xuICAgICAgICAgICAgICAgICR0aGlzLmZpbmQoJ3Rib2R5JykuYXR0cignZGF0YS1neC13aWRnZXQnLCAnY2hlY2tib3gnKTtcbiAgICAgICAgICAgICAgICBneC53aWRnZXRzLmluaXQoJHRoaXMpOyAvLyBJbml0aWFsaXplIHRoZSBjaGVja2JveCB3aWRnZXQuXG4gICAgICAgICAgICB9KTtcblxuICAgICAgICAgICAgLy8gQWRkIHNwaW5uZXIgdG8gdGFibGUgbG9hZGluZyBhY3Rpb25zLlxuICAgICAgICAgICAgdmFyICRzcGlubmVyO1xuICAgICAgICAgICAgJHRoaXMub24oJ3ByZVhoci5kdCcsIGZ1bmN0aW9uIChlLCBzZXR0aW5ncywganNvbikge1xuICAgICAgICAgICAgICAgICRzcGlubmVyID0ganNlLmxpYnMubG9hZGluZ19zcGlubmVyLnNob3coJHRoaXMpO1xuICAgICAgICAgICAgfSk7XG4gICAgICAgICAgICAkdGhpcy5vbigneGhyLmR0JywgZnVuY3Rpb24gKGUsIHNldHRpbmdzLCBqc29uKSB7XG4gICAgICAgICAgICAgICAgaWYgKCRzcGlubmVyKSB7XG4gICAgICAgICAgICAgICAgICAgIGpzZS5saWJzLmxvYWRpbmdfc3Bpbm5lci5oaWRlKCRzcGlubmVyKTtcbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICB9KTtcblxuICAgICAgICAgICAgLy8gQmluZCBldmVudCBoYW5kbGVycyBvZiB0aGUgZW1haWxzIHRhYmxlLlxuICAgICAgICAgICAgJHRoaXNcbiAgICAgICAgICAgICAgICAub24oJ2NsaWNrJywgJyNzZWxlY3QtYWxsLXJvd3MnLCBfb25TZWxlY3RBbGxSb3dzKVxuICAgICAgICAgICAgICAgIC5vbignY2xpY2snLCAnLnNlbmQtZW1haWwnLCBfb25TZW5kRW1haWwpXG4gICAgICAgICAgICAgICAgLm9uKCdjbGljaycsICcuZm9yd2FyZC1lbWFpbCcsIF9vbkZvcndhcmRFbWFpbClcbiAgICAgICAgICAgICAgICAub24oJ2NsaWNrJywgJy5kZWxldGUtZW1haWwnLCBfb25EZWxldGVFbWFpbClcbiAgICAgICAgICAgICAgICAub24oJ2NsaWNrJywgJy5wcmV2aWV3LWVtYWlsJywgX29uUHJldmlld0VtYWlsKTtcblxuICAgICAgICAgICAgZG9uZSgpO1xuICAgICAgICB9O1xuXG4gICAgICAgIC8vIFJldHVybiBkYXRhIHRvIG1vZHVsZSBlbmdpbmUuXG4gICAgICAgIHJldHVybiBtb2R1bGU7XG4gICAgfSk7XG4iXX0=
